classdef MVN
    %MVN = MultiVariate Normal distribution
    %
    % properties
    %
    %   N       degree of the random vector
    %   mu      mean (N-column vector)
    %   C       Covariance
    %   Ci2     inverse(C)/2
    %   d       denominator that divide by in computing pdf
    %
    % constructors
    %
    % mvn = MVN(C,mu)
    %   mu          N dimensional column vector or vct3
    %   C           N x N covariance matrix
    %
    % other methods
    %    % s = pdf(mvn,X)
    %   mvn         MVN corresponding to random N-vector X
    %   X           N dimensional column vector or vct3
    %   returns     probability density corresponding to X
    %
    % mNew = affine(mvn,M,b)
    %   mvn         MVN corresponding to random N-vector X
    %   M           k by k matrix
    %   b           k vector
    %   returns     MVN corresponding to M*X+b
    %
    % mNew = affine(mvn,F)
    %   returns     affine(mvn,F.R.el,F.p.el)
    %
    % mNew = affine(mvn,M)
    %   returns     affine(mvn,M,zeros(k,1))
    % 
    % Copyright (C) Russell H. Taylor 2013
    % For use with CIS I only
    % Do not redistribute without written permission from Russell Taylor

    
    properties
        N       % degree of the random vector
        mu      % mean
        C       % Covariance
        Ci2     % inverse cov/2
        d       % denominator that divide by in computing pdf 
     end
    
    methods
        function mvn = MVN(Cmx,mu)
            mvn.C = Cmx;
            if isa(mu,'vct3')
                mvn.mu = mu.el;
                N = 3;
            else
                mvn.mu = mu;
                N = size(mvn.mu,1);
                if size(mvn.mu,2)~=1
                    error('mean must be vct3 or column matrix');
                end                
            end
            if size(Cmx) ~= [N N]
                error('incompatible sizes for MVN');
            end
            mvn.N = N;
            [Q,D] = eig(Cmx);
            Ci = Q*diag(arrayfun(@(x)1/x,diag(D)))*Q';
            mvn.Ci2 = -Ci/2;
            mvn.d = (2*pi)^(N/2)*sqrt(det(D));
        end
        
        function p = pdf(mvn,X)
            if isa(X,'vct3')
                E=X.el-mvn.mu;
            else
                E=X-mvn.mu;
            end
            p = exp(E'*mvn.Ci2*E)/mvn.d;
        end
           
        function mNew = affine(mvn,M,b)
            % mNew = affine(mvn,M,b)
            %   mvn       MVN corresponding to random N-vector X
            %   M         k by k matrix
            %   b         k vector
            %   returns   MVN corresponding to M*X+b
            % mNew = affine(mvn,F)
            %   returns   affine(mvn,F.R.el,F.p.el)
            % mNew = affine(mvn,M)
            %   returns   affine(mvn,M,zeros(k,1))
            switch nargin
                case 2
                    if isa(M,'Frame')
                        mNew = affine(mvn,M.R.el,M.p);
                    else
                         mNew = affine(mvn,zeros(size(M,1)),M);
                    end
                case 3
                    mNew = MVN(M*mvn.C*M',M*mvn.mu);
                otherwise
                    error('Illegal number of arguments to MVN.affine')
            end
                    
        end
        
        function disp(mvn)
            disp('MVN');
            disp('  Covariance');
            disp(mvn.C);
            display('  Mean');
            disp(mvn.mu);
        end
        
        function display(mvn)
            disp(mvn);
        end
            
    end
    
end

